import React, { useCallback, useRef, useState } from "react";
import { View, Text, Image, TouchableOpacity } from "react-native";
import { styles } from "./reCaptchaStyle";
import ActivityIndicator from "./ActivityIndicator";
import { useSelector, useDispatch } from "react-redux";
import Recaptcha from "react-native-recaptcha-that-works";
import { storeRecaptchaToken } from "../../../redux/slices/auth/signIn";

import { RECAPTCHA_SITE_KEY, RECAPTCHA_BASE_URL } from "@env";
import useLangTranslation from "../../../hooks/useLangTranslation";
import useCustomToast from "../../../hooks/useCustomToast";

const LoginRecaptchaCheckbox = ({ reCaptchaError }) => {
    const { trans } = useLangTranslation();
    const { reCaptchaToken } = useSelector((state) => state.signInReducer);
    const [loading, setLoading] = useState(false);
    const dispatch = useDispatch();
    const recaptcha = useRef();
    const showToast = useCustomToast();

    const send = useCallback(() => {
        setLoading(true);
        recaptcha.current.open();
      }, []);

    const onVerify = (token) => {
        if(Boolean(token)){
            dispatch(storeRecaptchaToken(true));
        }
        setLoading(false);
    };
    return (
        <View>
            <Recaptcha
                lang='en'
                ref={recaptcha}
                size='invisible'
                hideBadge={true}
                onVerify={onVerify}
                baseUrl={RECAPTCHA_BASE_URL}
                siteKey={RECAPTCHA_SITE_KEY}
                onError={() => {
                    showToast({
                        text1: trans("Captcha error! try again later or contact site admin"),
                        type: 'common',
                        position: 'bottom',
                        props: { variant: 'error' }
                    });
                    setLoading(false);
                }}
                onClose={() => {
                    setLoading(false);
                }}
                style={styles.bgTransparent}
                loadingComponent={<Text></Text>}
            />
            <View style={styles.reCaptchaButton}>
                <View style={styles.reCaptchaText}>
                    <TouchableOpacity onPress={send} disabled={reCaptchaToken}>
                        <ActivityIndicator
                            reCaptchaToken={reCaptchaToken}
                            loading={loading}
                        />
                    </TouchableOpacity>
                    <Text>{trans("I'm not a robot")}</Text>
                </View>
                <View style={styles.reCaptchaIconWrap}>
                    <Image
                        style={styles.reCaptchaIcon}
                        source={require("../../../assets/images/reCaptcha.png")}
                    />
                    <Text style={styles.reCaptchaIconText}>reCAPTCHA</Text>
                </View>
            </View>
            {reCaptchaError && (
                <Text style={styles.reCaptchaError}>
                   {trans("Please verify that you are not a robot.")}
                </Text>
            )}
        </View>
    );
};

export default LoginRecaptchaCheckbox;
